/*
 *  $Id: scars.c 28910 2025-11-24 18:11:09Z yeti-dn $
 *  Copyright (C) 2003-2021 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>
#include "preview.h"

#define MARK_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)
#define REMOVE_RUN_MODES GWY_RUN_IMMEDIATE

enum {
    MAX_LENGTH = 1024
};

enum {
    PARAM_TYPE,
    PARAM_THRESHOLD_HIGH,
    PARAM_THRESHOLD_LOW,
    PARAM_MIN_LENGTH,
    PARAM_MAX_WIDTH,
    PARAM_COMBINE_TYPE,
    PARAM_COMBINE,
    PARAM_UPDATE,
    PARAM_MASK_COLOR,
};

typedef struct {
    GwyParams *params;
    GwyField *field;
    GwyField *mask;
    GwyField *result;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
} ModuleGUI;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             scars_remove        (GwyFile *data,
                                             GwyRunModeFlags mode);
static void             scars_mark          (GwyFile *data,
                                             GwyRunModeFlags mode);
static void             execute             (ModuleArgs *args);
static GwyDialogOutcome run_gui             (ModuleArgs *args,
                                             GwyFile *data,
                                             gint id);
static void             param_changed       (ModuleGUI *gui,
                                             gint id);
static void             preview             (gpointer user_data);
static void             sanitize_params     (GwyParams *params);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Marks and/or removes scars (horizontal linear artifacts)."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, scars)

static gboolean
module_register(void)
{
    gwy_process_func_register("scars_mark",
                              &scars_mark,
                              N_("/_Correct Data/M_ark Scars..."),
                              GWY_ICON_MARK_SCARS,
                              MARK_RUN_MODES,
                              GWY_MENU_FLAG_IMAGE,
                              N_("Mark horizontal scars (strokes)"));
    gwy_process_func_register("scars_remove",
                              &scars_remove,
                              N_("/_Correct Data/Remove _Scars"),
                              GWY_ICON_SCARS,
                              REMOVE_RUN_MODES,
                              GWY_MENU_FLAG_IMAGE,
                              N_("Correct horizontal scars (strokes)"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum feature_types[] = {
        { N_("Positive"), GWY_SIGN_POSITIVE, },
        { N_("Negative"), GWY_SIGN_NEGATIVE, },
        { N_("Both"),     GWY_SIGN_BOTH,     },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, "scars");
    gwy_param_def_add_gwyenum(paramdef, PARAM_TYPE, "type", _("Scars type"),
                              feature_types, G_N_ELEMENTS(feature_types), GWY_SIGN_BOTH);
    gwy_param_def_add_double(paramdef, PARAM_THRESHOLD_HIGH, "threshold_high", _("_Hard threshold"), 0.0, 2.0, 0.666);
    gwy_param_def_add_double(paramdef, PARAM_THRESHOLD_LOW, "threshold_low", _("_Soft threshold"), 0.0, 2.0, 0.25);
    gwy_param_def_add_int(paramdef, PARAM_MIN_LENGTH, "min_len", _("Minimum _length"), 1, MAX_LENGTH, 16);
    gwy_param_def_add_int(paramdef, PARAM_MAX_WIDTH, "max_width", _("Maximum _width"), 1, 16, 4);
    gwy_param_def_add_enum(paramdef, PARAM_COMBINE_TYPE, "combine_type", NULL, GWY_TYPE_MERGE_TYPE, GWY_MERGE_UNION);
    gwy_param_def_add_boolean(paramdef, PARAM_COMBINE, "combine", NULL, FALSE);
    gwy_param_def_add_instant_updates(paramdef, PARAM_UPDATE, "update", NULL, TRUE);
    gwy_param_def_add_mask_color(paramdef, PARAM_MASK_COLOR, NULL, NULL);
    return paramdef;
}

static void
mark_scars(GwyField *field, GwyField *mask,
           GwyParams *params)
{
    GwySignFlags type = gwy_params_get_enum(params, PARAM_TYPE);
    gdouble threshold_high = gwy_params_get_double(params, PARAM_THRESHOLD_HIGH);
    gdouble threshold_low = gwy_params_get_double(params, PARAM_THRESHOLD_LOW);
    gint min_len = gwy_params_get_int(params, PARAM_MIN_LENGTH);
    gint max_width = gwy_params_get_int(params, PARAM_MAX_WIDTH);
    GwyField *tmp;

    if (type == GWY_SIGN_POSITIVE || type == GWY_SIGN_NEGATIVE) {
        gwy_field_mark_scars(field, mask, threshold_high, threshold_low,
                             min_len, max_width, type == GWY_SIGN_NEGATIVE);
        return;
    }

    gwy_field_mark_scars(field, mask, threshold_high, threshold_low, min_len, max_width, FALSE);
    tmp = gwy_field_new_alike(field, FALSE);
    gwy_field_mark_scars(field, tmp, threshold_high, threshold_low, min_len, max_width, TRUE);
    gwy_field_max_of_fields(mask, mask, tmp);
    g_object_unref(tmp);
}

static void
scars_remove(GwyFile *data, GwyRunModeFlags mode)
{
    GwyParams *params;
    GwyField *field, *mask;
    GQuark dquark;
    gint id;

    g_return_if_fail(mode & REMOVE_RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD_KEY, &dquark,
                                 GWY_APP_FIELD, &field,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(field && dquark);

    params = gwy_params_new_from_settings(define_module_params());
    sanitize_params(params);
    gwy_app_undo_qcheckpointv(GWY_CONTAINER(data), 1, &dquark);

    mask = gwy_field_new_alike(field, TRUE);
    gwy_unit_clear(gwy_field_get_unit_z(mask));
    mark_scars(field, mask, params);
    gwy_field_laplace_solve(field, mask, -1, 1.0);
    g_object_unref(mask);
    g_object_unref(params);

    gwy_field_data_changed(field);
    gwy_log_add_full(data, GWY_FILE_IMAGE, id, id, "proc::scars_remove",
                     "settings-name", "scars",
                     NULL);
}

static void
scars_mark(GwyFile *data, GwyRunModeFlags mode)
{
    ModuleArgs args;
    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    GQuark mquark;
    gint id;

    g_return_if_fail(mode & MARK_RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD, &args.field,
                                 GWY_APP_MASK_FIELD_KEY, &mquark,
                                 GWY_APP_MASK_FIELD, &args.mask,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(args.field && mquark);

    args.result = gwy_field_new_alike(args.field, TRUE);
    gwy_unit_clear(gwy_field_get_unit_z(args.result));
    args.params = gwy_params_new_from_settings(define_module_params());
    sanitize_params(args.params);
    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, data, id);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (outcome != GWY_DIALOG_HAVE_RESULT)
        execute(&args);

    gwy_app_undo_qcheckpointv(GWY_CONTAINER(data), 1, &mquark);
    if (gwy_field_get_max(args.result) > 0.0)
        gwy_file_set_image_mask(data, id, args.result);
    else
        gwy_container_remove(GWY_CONTAINER(data), mquark);
    gwy_log_add_full(data, GWY_FILE_IMAGE, id, id, "proc::scars_remove",
                     "settings-name", "scars",
                     NULL);

end:
    g_object_unref(args.result);
    g_object_unref(args.params);
}

static void
execute(ModuleArgs *args)
{
    gboolean combine = gwy_params_get_boolean(args->params, PARAM_COMBINE);
    GwyMergeType combine_type = gwy_params_get_enum(args->params, PARAM_COMBINE_TYPE);

    mark_scars(args->field, args->result, args->params);
    if (args->mask && combine) {
        if (combine_type == GWY_MERGE_UNION)
            gwy_field_grains_add(args->result, args->mask);
        else if (combine_type == GWY_MERGE_INTERSECTION)
            gwy_field_grains_intersect(args->result, args->mask);
    }
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, GwyFile *data, gint id)
{
    GtkWidget *hbox, *dataview;
    GwyDialog *dialog;
    GwyParamTable *table;
    ModuleGUI gui;

    gwy_clear1(gui);
    gui.args = args;

    gui.dialog = gwy_dialog_new(_("Mark Scars"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_UPDATE, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    dataview = gwy_create_preview(args->field, args->result, PREVIEW_SIZE);
    gwy_setup_data_view(GWY_DATA_VIEW(dataview), data, GWY_FILE_IMAGE, id,
                        GWY_FILE_ITEM_PALETTE | GWY_FILE_ITEM_MASK_COLOR
                        | GWY_FILE_ITEM_RANGE | GWY_FILE_ITEM_REAL_SQUARE);
    hbox = gwy_create_dialog_preview_hbox(GTK_DIALOG(dialog), GWY_DATA_VIEW(dataview), FALSE);

    table = gui.table = gwy_param_table_new(args->params);

    gwy_param_table_append_slider(table, PARAM_MAX_WIDTH);
    gwy_param_table_set_unitstr(table, PARAM_MAX_WIDTH, _("px"));
    gwy_param_table_slider_set_mapping(table, PARAM_MAX_WIDTH, GWY_SCALE_MAPPING_LINEAR);
    gwy_param_table_append_slider(table, PARAM_MIN_LENGTH);
    gwy_param_table_set_unitstr(table, PARAM_MIN_LENGTH, _("px"));

    gwy_param_table_append_slider(table, PARAM_THRESHOLD_HIGH);
    gwy_param_table_set_unitstr(table, PARAM_THRESHOLD_HIGH, _("RMS"));
    gwy_param_table_append_slider(table, PARAM_THRESHOLD_LOW);
    gwy_param_table_set_unitstr(table, PARAM_THRESHOLD_LOW, _("RMS"));

    gwy_param_table_append_separator(table);
    gwy_param_table_append_radio(table, PARAM_TYPE);

    gwy_param_table_append_header(table, -1, _("Options"));
    gwy_param_table_append_image_mask_color(table, PARAM_MASK_COLOR, GWY_DATA_VIEW(dataview), data, id);
    if (args->mask) {
        gwy_param_table_append_radio_buttons(table, PARAM_COMBINE_TYPE, NULL);
        gwy_param_table_add_enabler(table, PARAM_COMBINE, PARAM_COMBINE_TYPE);
    }
    gwy_param_table_append_checkbox(table, PARAM_UPDATE);

    gtk_box_pack_start(GTK_BOX(hbox), gwy_param_table_widget(table), TRUE, TRUE, 0);
    gwy_dialog_add_param_table(dialog, table);

    g_signal_connect_swapped(table, "param-changed", G_CALLBACK(param_changed), &gui);
    gwy_dialog_set_preview_func(dialog, GWY_PREVIEW_IMMEDIATE, preview, &gui, NULL);

    return gwy_dialog_run(dialog);
}

static void
param_changed(ModuleGUI *gui, gint id)
{
    GwyParams *params = gui->args->params;
    GwyParamTable *table = gui->table;

    if (id == PARAM_THRESHOLD_HIGH || id == PARAM_THRESHOLD_LOW) {
        gdouble threshold_low = gwy_params_get_double(params, PARAM_THRESHOLD_LOW);
        gdouble threshold_high = gwy_params_get_double(params, PARAM_THRESHOLD_HIGH);
        if (threshold_high < threshold_low) {
            if (id == PARAM_THRESHOLD_HIGH)
                gwy_param_table_set_double(table, PARAM_THRESHOLD_LOW, threshold_high);
            else
                gwy_param_table_set_double(table, PARAM_THRESHOLD_HIGH, threshold_low);
        }
    }
    if (id != PARAM_MASK_COLOR && id != PARAM_UPDATE)
        gwy_dialog_invalidate(GWY_DIALOG(gui->dialog));
}

static void
preview(gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    ModuleArgs *args = gui->args;

    execute(args);
    gwy_field_data_changed(args->result);
    gwy_dialog_have_result(GWY_DIALOG(gui->dialog));
}

static void
sanitize_params(GwyParams *params)
{
    gdouble threshold_high = gwy_params_get_double(params, PARAM_THRESHOLD_HIGH);
    gdouble threshold_low = gwy_params_get_double(params, PARAM_THRESHOLD_LOW);

    if (threshold_low > threshold_high)
        gwy_params_set_double(params, PARAM_THRESHOLD_HIGH, threshold_low);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
