/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2026 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - MassXpert, model polymer chemistries and simulate mass spectrometric data;
 * - MineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes
#include <QFile>
#include <QDebug>


/////////////////////// Local includes
#include "MonomerSpec.hpp"


namespace MsXpS
{
namespace MassXpert
{


void
MonomerSpec::setName(const QString &name)
{
  m_name = name;
}


const QString &
MonomerSpec::getName() const
{
  return m_name;
}

void
MonomerSpec::setCode(const QString &code)
{
  m_code = code;
}


const QString &
MonomerSpec::getCode() const
{
  return m_code;
}


void
MonomerSpec::setRaster(const QString &raster)
{
  m_raster = raster;
}


const QString &
MonomerSpec::getRaster() const
{
  return m_raster;
}


void
MonomerSpec::setVector(const QString &vector)
{
  m_vector = vector;
}


const QString &
MonomerSpec::getVector() const
{
  return m_vector;
}


void
MonomerSpec::setNameSound(const QString &sound)
{
  m_nameSound = sound;
}


const QString &
MonomerSpec::getNameSound() const
{
  return m_nameSound;
}


void
MonomerSpec::setCodeSound(const QString &sound)
{
  m_codeSound = sound;
}


const QString &
MonomerSpec::getCodeSound() const
{
  return m_codeSound;
}


bool
MonomerSpec::parseFile(const QString &file_path,
                       std::vector<MonomerSpec> &monomer_specs)
{
  MonomerSpec monomer_spec;

  qint64 lineLength;

  QString line;
  QString temp;

  char buffer[1024];

  int percentSignIdx = 0;
  int pipeSignIdx    = 0;

  if(file_path.isEmpty())
    return false;

  QFile file(file_path);

  if(!file.open(QFile::ReadOnly))
    return false;

  // The lines we have to parse are of the following type:
  // A%alanine.svg|alanine.png
  // Any line starting with ' ' or '#' are not parsed.

  // Get the first line of the file. Next we enter in to a
  // while loop.

  lineLength = file.readLine(buffer, sizeof(buffer));

  while(lineLength != -1)
    {
      // The line is now in buffer, and we want to convert
      // it to Unicode by setting it in a QString.
      line = buffer;

      // The line that is in line should contain something like:
      // A%alanine.svg|alanine.png
      //
      // Note, however that lines beginning with either ' '(space) or
      // '\n'(newline) or '#' are comments.

      // Remove all the spaces from the borders: Whitespace means any
      // character for which QChar::isSpace() returns true. This
      // includes the ASCII characters '\t', '\n', '\v', '\f', '\r',
      // and ' '.

      line = line.trimmed();

      if(line.isEmpty() || line.startsWith('#', Qt::CaseInsensitive))
        {
          lineLength = file.readLine(buffer, sizeof(buffer));
          continue;
        }

      // Now some other checks. Remember the format of the line:
      // A%alanine.svg|alanine.png

      percentSignIdx = line.indexOf('%', 0, Qt::CaseInsensitive);
      pipeSignIdx    = line.indexOf('|', 0, Qt::CaseInsensitive);

      if(percentSignIdx == -1 || line.count('%', Qt::CaseInsensitive) > 1 ||
         line.count('|', Qt::CaseInsensitive) > 1)
        return false;

      if(pipeSignIdx != -1)
        if(percentSignIdx > pipeSignIdx)
          return false;

      monomer_spec.m_code = line.left(percentSignIdx);

      // Remove from the line the "A%" substring, as we do not need it
      // anymore.
      line.remove(0, percentSignIdx + 1);
      if(pipeSignIdx == -1)
        {
          // If there was no pipe sign in the line, we have a single
          // substring up to the end of the line.

          if(line.endsWith(".png", Qt::CaseSensitive))
            monomer_spec.m_raster = line;
          else if(line.endsWith(".svg", Qt::CaseSensitive))
            monomer_spec.m_vector = line;
          else
            return false;
        }
      else
        {
          // There was a pipe sign indeed, we have to divide the
          // remaining substring into two parts, before and after the
          // '|'.

          // Extract the left substring.
          temp = line.section('|', 0, 0);
          if(temp.endsWith(".png", Qt::CaseSensitive))
            monomer_spec.m_raster = temp;
          else if(temp.endsWith(".svg", Qt::CaseSensitive))
            monomer_spec.m_vector = temp;
          else
            return false;


          // Extract the right substring.
          temp = line.section('|', 1, 1);
          if(temp.endsWith(".png", Qt::CaseSensitive))
            monomer_spec.m_raster = temp;
          else if(temp.endsWith(".svg", Qt::CaseSensitive))
            monomer_spec.m_vector = temp;
          else
            return false;
        }

      monomer_specs.push_back(monomer_spec);
      lineLength = file.readLine(buffer, sizeof(buffer));
    }

  return true;
}


} // namespace MassXpert
} // namespace MsXpS
