/*****************************************************************************
 * Media Library
 *****************************************************************************
 * Copyright (C) 2015-2019 Hugo Beauzée-Luyssen, Videolabs, VideoLAN
 *
 * Authors: Hugo Beauzée-Luyssen <hugo@beauzee.fr>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *****************************************************************************/

#ifndef AUDIOTRACK_H
#define AUDIOTRACK_H

#include "medialibrary/IAudioTrack.h"
#include "database/DatabaseHelpers.h"

namespace medialibrary
{

class AudioTrack : public IAudioTrack, public DatabaseHelpers<AudioTrack>
{
    public:
        struct Table
        {
            static const std::string Name;
            static const std::string PrimaryKeyColumn;
            static int64_t AudioTrack::* const PrimaryKey;
        };
        enum class Indexes : uint8_t
        {
            MediaId,
            AttachedFileId,
        };

        AudioTrack(MediaLibraryPtr ml, sqlite::Row& row );
        AudioTrack( MediaLibraryPtr ml, std::string codec, unsigned int bitrate,
                    unsigned int sampleRate, unsigned int nbChannels, std::string language,
                    std::string desc, int64_t mediaId, int64_t attachedFileId );

        virtual int64_t id() const override;
        virtual const std::string&codec() const override;
        virtual unsigned int bitrate() const override;
        virtual unsigned int sampleRate() const override;
        virtual unsigned int nbChannels() const override;
        virtual const std::string& language() const override;
        virtual const std::string& description() const override;
        virtual bool isInAttachedFile() const override;

        static void createTable( sqlite::Connection* dbConnection );
        static void createIndexes( sqlite::Connection* dbConnection );
        static std::string schema( const std::string& tableName, uint32_t dbModel );
        static std::string index( Indexes index, uint32_t dbModel );
        static std::string indexName( Indexes index, uint32_t dbModel );
        static bool checkDbModel( MediaLibraryPtr ml );
        static std::shared_ptr<AudioTrack> create( MediaLibraryPtr ml, std::string codec,
                                                   unsigned int bitrate, unsigned int sampleRate, unsigned int nbChannels,
                                                   std::string language, std::string desc,
                                                   int64_t mediaId, int64_t attachedFileId );
        static bool removeFromMedia( MediaLibraryPtr ml, int64_t mediaId,
                                     bool internalTracksOnly );
        static Query<IAudioTrack> fromMedia( MediaLibraryPtr ml, int64_t mediaId,
                                             bool internalTracksOnly );

    private:
        int64_t m_id;
        const std::string m_codec;
        const unsigned int m_bitrate;
        const unsigned int m_sampleRate;
        const unsigned int m_nbChannels;
        const std::string m_language;
        const std::string m_description;
        const int64_t m_mediaId;
        const int64_t m_attachedFileId;

    private:
        friend struct AudioTrack::Table;
};

}

#endif // AUDIOTRACK_H
